
=head1 NAME

DirwalkCallback

=head1 SYNOPSIS

  use DirwalkCallback;
  my $print_pl_only = sub { 
      my $f = shift;
      print $f, "\n"  if $f =~ m/\.pl$/;
  };

  # Get text files from home directory (but not subdirs)
  dirwalk_with_callback( dir => $ENV{HOME},
      func => $print_pl_only );

  # Get text files from home directory and subdirs
  dirwalk_with_callback( dir => $ENV{HOME},
      func => $print_pl_only,
      recurse => 1, );

  # Get text files from home directory and subdirs and
  # be chatty about it
  dirwalk_with_callback( dir => $ENV{HOME},
      func => $print_pl_only,
      recurse => 1,
      verbose => 1 );

=head1 DESCRIPTION

Basically a poor man's C<File::Find>, except that it doesn't recurse
by default.  As far as I could tell, File::Find doesn't have an option
to not recurse.


=head1 DEPENDENCIES

Requires Perl 5.10 or higher.

C<warnings>, C<strict> and C<Carp> are all in the standard library.

=cut


package DirwalkCallback;

use warnings;
use strict;
use Carp;
use v5.10;

our $VERSION = "1.0";

BEGIN {
    use Exporter();
    our @ISA = qw(Exporter);

    our @EXPORT = qw( 
	dirwalk_with_callback
	);

    return 1;
}


=head1 dirwalk_with_callback( options hash )

Takes a hash of options.  The available options are:

=over

=item dir

The directory to walk.

=item func

A reference to a callback function that is called for every
regular file (files for which the -f filetest returns true).
It gets a full path to the file as its sole argument.

=item recurse

If set to nonzero, we'll recurse into subdirectories.

=item verbose

If set to nonzero, we'll print messages as we work through 
the directory.

=back

=cut



###TODO rewrite this to use a queue rather than recurse?
sub dirwalk_with_callback {
    if ( scalar @_ % 2 == 1 ) {
	croak "dirwalk_with_callback() called with odd number of options";
    }

    my %options = @_;    
    my @valid_opts = qw( dir func verbose recurse );
    for my $opt ( keys %options ) {
	unless ( grep { $opt eq $_ } @valid_opts ) {
	    croak "Invalid option '$opt' passsed to to dirwalk_with_callback()";
	}
    }

#    print join ", ", map { defined $options{$_} ? "$_ : $options{$_}" : "$_ : undefined" } ( sort keys %options);
#    print "\n";
    
    if ( not $options{dir} or not -e $options{dir} or not -d $options{dir} ) {
	croak "dir argument of dirwalk_with_callback() must be a valid directory\n";
    }
    if ( not $options{func} or ref $options{func} ne 'CODE' ) {
	croak "func argument of dirwalk_with_callback() must be a function reference\n";
    }

    $options{dir} =~ s!/$!!;	# remove trailing slash. doesn't matter for
				# functionality but makes for more readable
				# verbose output.
    
    my $dh;
    opendir $dh, $options{dir}	or die;
    say "opened $options{dir}" if $options{verbose};

    while ( my $f = readdir $dh ) {
	my $fullpath = $options{dir} . "/" . $f;
	if ( -f $fullpath ) {
	    $options{func}->( $fullpath );
	} elsif ( -d $fullpath ) {
	    next unless $options{recurse};
	    unless ( $f eq '.' or $f eq '..' ) {
	        say "$f is a directory, recursing" if $options{verbose};
		dirwalk_with_callback( dir => $fullpath,
				       func => $options{func},
				       recurse => 1,
				       verbose => $options{verbose} );
	    }
	} else {
	    # probably a device or symlink or something we don't care about	
	    say "$f is neither a plain file or directory" if $options{verbose};
	}
    }
    say "finished with $options{dir}"  if $options{verbose};

}

return 1;

=head1 AUTHOR

Jim Henry III, L<http://jimhenry.conlang.org/software/>


=head1 LICENSE

This script is free software; you may redistribute it and/or modify it under
the same terms as Perl itself.

=cut
