#! /usr/bin/perl -w

=head1 NAME 

stegspace.pl

=head1 SYNOPSIS

  # Hide a message in a text file.
  stegspace.pl --message Hi! --haystack README

  # Hide a smaller text file in a larger text file.
  stegspace.pl --needle passwords.txt --haystack Bible.txt

  # Retrieve the messages.
  stegspace.pl --haystack README
  stegspace.pl --haystack Bible.txt

  # Print an error message and exit if there are not enough lines in the larger text
  # file to hide the secret text without padding the end with "blank" lines.
  stegspace.pl --fail  --needle Bible.txt --haystack passwords.txt


=head1 DESCRIPTION

This script inserts or reads a steganographic message in a text file, encoded as variable
number of spaces at ends of lines.  Each line can have 0-3 spaces at the end,
encoding two bits, so one byte of the message requires four lines of the
haystack file.  It keeps the access and modification times of the haystack file the
same as before.

For reference, a 5k text file hidden in the Bible (Project Gutenberg
Douay-Rheims Version) adds spaces to the ends of lines as far as Deuteronomy
15:12, about 20k lines.


=head1 DISCLAIMER

This is just for fun, not adequate security for actually concealing passwords
as in the L</SYNOPSIS>.


=head1 OPTIONS

=over

=item -m  --message

Message to hide in the haystack file.

=item -n  --needle

Text file to hide in the haystack file.

=item -h  --haystack

Large text file in which to hide a short message or text file.

=item -f  --fail

Exit immediately if a warning condition occurs (e.g., not enough lines in the haystack file)

=item --help

Get brief help.

=item -M --manual

Get detailed help.

=item -d  --debug

Print internal diagnostic and progress messages 

=back


=head1 DEPENDENCIES

C<warnings>, C<strict>, C<constant>, C<Pod::Usage>, and C<Getopt::Long> are all
in the standard library.


=head1 LIMITATIONS

It can't encode a message that includes null bytes, because it uses a zero byte
(four consecutive lines with no trailing spaces) as an end of message marker.

I could remove that limitation by starting with a few lines encoding message
length instead, but it's probably not worth it for a toy script like this.



=head1 AUTHOR

Jim Henry III, L<http://jimhenry.conlang.org/software/>


=head1 LICENSE

This script is free software; you may redistribute it and/or modify it under
the same terms as Perl itself.

=cut

use strict;
use warnings;
use Pod::Usage;
use Getopt::Long;
Getopt::Long::Configure('bundling');


use constant READ => 97;
use constant WRITE => 98;
my $version = 2;
my $debug = 0;
my $mode;
my $needle_file;
my $needle_string;
my $haystack_file;
my $fail_on_warning = 0;

sub help {
	my $name = $0;
	$name =~ s(.*/)();
	print<<HELP;

$name version $version

Usage:

-h  --haystack=[filename]       Large text file in which to hide a short message or text file.
-m  --message=[string]          Message to hide in the haystack file.
-n  --needle=[filename]         Text file to hide in the haystack file.
-f  --fail                      Exit immediately if a warning condition occurs 
				(e.g., not enough lines in the haystack file)
    --help			Get brief help.
-M  --manual			Get detailed help.
-d  --debug                    	Print internal diagnostic and progress messages 

To hide a message, use e.g. 

$name --message Hi! --haystack README
$name --needle passwords.txt --haystack Bible.txt

To retrieve a message, use

$name --haystack Bible.txt

HELP

}


sub warnmsg {
    warn shift;
    exit if $fail_on_warning;
}


my $help = 0;
my $manual = 0;

if ( not GetOptions(
	'm|message=s' 	=> \$needle_string,
	'n|needle=s'	=> \$needle_file,
    	'h|haystack=s' 	=> \$haystack_file,
	'd|debug'	=> \$debug,
	'f|fail'        => \$fail_on_warning,
        'help'          => \$help,
        'M|manual'      => \$manual,
     ) )
{
    help;
    exit(1);
}

if ( $help ) {
    help;
    exit(0);
} elsif ( $manual ) {
    pod2usage(-verbose => 2);
    exit(0);
}


if ( not defined $needle_string  and  not defined $needle_file ) {
    $mode = READ;
} elsif ( defined  $needle_string ) {
    $mode = WRITE;
} else {
    $mode = WRITE;
    open IN, $needle_file	or die "can't open $needle_file for reading\n";
    my @lines = <IN>;
    $needle_string = join "", @lines;
    close IN;
}

print STDERR "needle string is " . ( length $needle_string ) . " chars\n"	if $debug;

if ( not defined $haystack_file ) {
    print "-h / --haystack argument is required\n";
    exit;
}

if ( $mode == WRITE ) {
    my $tmpfile = $haystack_file . ".tmp";  # TODO use File::Temp::tempfile instead
    my @stat_rc = stat($haystack_file);
    my ($atime, $mtime) = @stat_rc[ 8, 9 ];
    printf STDERR "$haystack_file access time %d modification time %d\n", $atime, $mtime	if $debug;

    open IN, $haystack_file;
    my @contents = <IN>;

    my $lines_needed = length($needle_string) * 4;
    if ( scalar @contents < $lines_needed ) {
	my $maxchrs = int(scalar @contents / 4);
	warnmsg "Not enough lines in $haystack_file.\n(We need a file of at least $lines_needed lines to avoid padding the end.\n" .
                "The biggest needle you could hide in this haystack would be $maxchrs characters.)\n";
	# if that function didn't exit:
	print "Going to pad the end with blank lines as needed.\n";
    }
    
    my $line_idx = 0;
    close IN;

    my @chars = split //, $needle_string;
    open OUT, ">" . $tmpfile;
    foreach ( @chars ) {
	my $n = ord $_;
	print STDERR "char $_ is ascii $n\n"	if $debug > 1;
	for ( my $i = 0;  $i <= 3;  $i++ ) {
	    my $bits = $n % 4;
	    print STDERR "bits: $bits\n" 	if $debug > 1;
	    $n = $n >> 2;

	    if ( $line_idx <= $#contents ) {
		# rewrite line of input file with adjusted end-line spacing
		# first strip any trailing spaces from  line as we found it
		$contents[ $line_idx ] =~ s/\s+$//;
		print OUT $contents[ $line_idx ] . (" " x $bits) . "\n";
		$line_idx++;
	    } else {
		# print one or more space-lines at end of file
		print OUT (" " x $bits) . "\n";
	    }
	}
    }
    while ( $line_idx <= $#contents ) {
	$contents[ $line_idx ] =~ s/\s+$//;
	print OUT $contents[ $line_idx++ ] . "\n";
    }
    close OUT;

    rename $tmpfile,  $haystack_file	        or die "can't overwrite $haystack_file with $tmpfile\n";
    utime( $atime, $mtime, $haystack_file )	or die "utime failed on $haystack_file\n";

} elsif ( $mode == READ ) {
    my $charnum = 0;
    my $n = 0;
    my @bits = (0, 0, 0, 0, 0);
    my $idx;
    open IN, $haystack_file;
    while ( <IN> ) {
	m/[^\s]*( *)$/;
	my $spaces = $1;
	$n = length $spaces;
	print STDERR "line $. spaces = $n\n"	if $debug > 1;
	$idx = ( $. % 4 == 0 ) ? 4 : ( $. % 4 ) ;
	$bits[ $idx ] = $n;
	
	if ( $. % 4 == 0 ) {
	    print STDERR ((( join ", ", @bits )) . "\n")	if $debug > 1;
	    for ( my $i = 4;  $i >= 1;  $i-- ) {
		$charnum = $charnum << 2;
		$charnum += $bits[ $i ];
	    }
	   
	    if ( $charnum == 0 ) {
		last;
	    }

	    my $char = chr $charnum;
	    print STDERR "line $. char = $charnum / [$char]\n"	if $debug > 1;
	    $needle_string .= $char;
	    $charnum = 0;
	}
    }
    close IN;
    if ( not $needle_string ) {
	warnmsg "No needle found in $haystack_file\n";
    } else {
	my $newline_needed = $needle_string =~ m/\n$/ ? '' : "\n";
	print $needle_string . $newline_needed;
    }
} else {
    die "bad value for \$mode == $mode\n";
}
